#include "precompiled.h"
#include "LensImporter.h"
#include "common.h"

#include "Camera.h"
#include "Lens.h"
#include "Aperture.h"
#include "Texture1D.h"

namespace RTCam {

const int kMaxRadiusFromFilmCenter = 22;

LensImporter::LensImporter(void)
{
}


LensImporter::~LensImporter(void)
{
}

bool LensImporter::ImportLens(_In_ ID3D11Device1* device, _In_ Camera* camera)
{
	bool success = false;

	// The file type filters
	COMDLG_FILTERSPEC fileTypes[] =
	{ 
		{ L"Text files", L"*.txt" },
		{ L"All files", L"*.*" },
	};

	string filepath = GetFilePath(fileTypes, ARRAYSIZE(fileTypes));
	if(filepath.length() > 0) {
		success = Import(device, filepath.c_str(), camera);
	}

	return success;
}

bool LensImporter::Import( _In_ ID3D11Device1* device, _In_z_ const char* filename, _In_ Camera* camera)
{
	const int WordLength = 100;

	std::ifstream filestream;
	filestream.open(filename, std::ios_base::in);
	
	if (!filestream.is_open()) {
		DebugPrint(StringFormat("Unable to open file %s\n", filename).c_str());
		return false;
	}

	string line;
	char buffer[WordLength];
	
	if(!CheckToken("LENS NAME", filestream)) return false;
	camera->m_currentFilmName = GetLine(filestream);

	//----- Parse focal length (mm) -----
	if(!CheckToken("FOCAL LENGTH", filestream)) return false;
	line = GetLine(filestream);
	
	// TODO: Support ranges of focal lengths for zoom lenses
	float focalLength;
	if(sscanf_s(line.c_str(), "%f", &focalLength) != 1) {
		DebugPrint("Parse error: Failed to parse focal length.\n");
		filestream.close();
		return false;
	}
	focalLength *= kMmToMeters;

	//----- Parse min focal distance (meters) -----
	if(!CheckToken("MIN FOCAL DISTANCE", filestream)) return false;
	line = GetLine(filestream);

	float minFocalDistance;
	if(sscanf_s(line.c_str(), "%f", &minFocalDistance) != 1) {
		DebugPrint("Parse error: Failed to parse min focal distance.\n");
		filestream.close();
		return false;
	}

	camera->m_lens->SetFocalRanges(focalLength, focalLength, minFocalDistance);
	
	//----- Parse aperture range -----
	if(!CheckToken("APERTURE RANGE", filestream)) return false;
	line = GetLine(filestream);
	
	float minFNumber, maxFNumber;
	if(sscanf_s(line.c_str(), "%f %f", &minFNumber, &maxFNumber) != 2) {
		DebugPrint("Parse error: Failed to parse f-Numbers.\n");
		filestream.close();
		return false;
	}
	camera->m_aperture->SetFNumberRange(minFNumber, maxFNumber);

	//----- Parse vignetting -----
	if(!CheckToken("RELATIVE ILLUMINANCE", filestream)) return false;
	line = GetLine(filestream);

	int numCharts;
	if(sscanf_s(line.c_str(), "%i", &numCharts) != 1) {
		DebugPrint("Parse error: Failed to parse the number of vignetting charts.\n");
		filestream.close();
		return false;
	}

	camera->m_lens->m_vignettingTexs.clear();

	for(int i = 0; i < numCharts; ++i) {
		line = GetLine(filestream);

		Lens::Vignetting v = {0};
		if(sscanf_s(line.c_str(), "%s %f", buffer, WordLength, &v.fNumber) != 2) {
			DebugPrint("Parse error: Failed to parse the vignetting f-Number.\n");
			filestream.close();
			return false;
		}

		v.vignettingTex = ReadChart(device, filestream, StringFormat("Vignetting %.1f Tex", v.fNumber).c_str());
		if(v.vignettingTex == nullptr) {
			DebugPrint("Parse error: Failed to parse the vignetting chart.\n");
			filestream.close();
			return false;
		}

		camera->m_lens->m_vignettingTexs.push_back(v);
	}

	//----- Parse distortion -----
	if(!CheckToken("RELATIVE DISTORTION", filestream)) return false;
	auto distortionTex = ReadChart(device, filestream, "Distortion Tex");

	camera->m_lens->m_distortionTex = distortionTex;

	//----- Parse field curvature -----
	if(!CheckToken("FIELD CURVATURE", filestream)) return false;
	auto fieldCurvatureTex = ReadChart(device, filestream, "Field Curvature Tex");

	// TODO: Implement

	//----- Parse astigmatism -----
	if(!CheckToken("ASTIGMATISM", filestream)) return false;
	if(!CheckToken("SAGITTAL", filestream)) return false;
	auto sagittalTex = ReadChart(device, filestream, "Sagittal Tex");
	if(!CheckToken("TANGENTIAL", filestream)) return false;
	auto tangentialTex = ReadChart(device, filestream, "Tangential Tex");

	camera->m_lens->m_sagittalTex = sagittalTex;
	camera->m_lens->m_tangentialTex = tangentialTex;

	//----- Parse spherical aberration -----
	if(!CheckToken("SPHERICAL ABERRATION", filestream)) return false;
	line = GetLine(filestream);
	
	float sphericalAberration;
	if(sscanf_s(line.c_str(), "%f", &sphericalAberration) != 1) {
		DebugPrint("Parse error: Failed to parse spherical aberration.\n");
		filestream.close();
		return false;
	}
	camera->m_lens->m_sphericalAberrationFactor = sphericalAberration;

	filestream.close();

	return true;
}

bool LensImporter::CheckToken(_In_z_ const char* token, std::ifstream& filestream)
{
	if(strcmp(GetLine(filestream).c_str(), token) != 0) {
		DebugPrint(StringFormat("Parse error: Expected to read token %s.\n", token).c_str());
		filestream.close();
		return false;
	}

	return true;
}

shared_ptr<Texture1D> LensImporter::ReadChart(_In_ ID3D11Device1* device, std::ifstream& filestream, _In_z_ const char* texName)
{
	int numValues;
	string line = GetLine(filestream);
	if(sscanf_s(line.c_str(), "%i", &numValues) != 1) {
		return nullptr;
	}

	const int arraySize = kMaxRadiusFromFilmCenter + 1;
	float chartValues[arraySize];
	
	int prevRadius = -1;
	int curRadius;
	float prevValue;
	float curValue;

	for(int i = 0; i < numValues; ++i) {
		line = GetLine(filestream);
		if(sscanf_s(line.c_str(), "%i %f", &curRadius, &curValue) != 2) {
			return nullptr;
		}

		// NOTE: File must start with a value for a radius of 0
		if(prevRadius == -1) {
			if(curRadius != 0) {
				return nullptr;
			}

			prevRadius = 0;
			chartValues[0] = curValue;
			prevValue = curValue;
		} else {
			// Make sure that the current radius value is valid and larger than the previous.
			if(curRadius <= prevRadius || curRadius >= arraySize) {
				return nullptr;
			}

			// Linearly interpolate and fill in values up to the current radius.
			float range = static_cast<float>(curRadius - prevRadius);
			while(prevRadius < curRadius) {
				++prevRadius;
				chartValues[prevRadius] = ClampedLerp(prevValue, curValue, 1 - ((curRadius - prevRadius) / range));
			}

			ASSERT(prevRadius == curRadius);
			prevValue = curValue;
		}
	}
	
	// Fill any remaining empty spots with the last read value
	++prevRadius;
	while(prevRadius < arraySize) {
		chartValues[prevRadius] = prevValue;
		++prevRadius;
	}

	shared_ptr<Texture1D> texture(new Texture1D(texName));
	texture->InitFromChart(device, chartValues, arraySize);

	return texture;
}

} // end namespace